/**
 *
 * \file        serialport.c
 *
 * \brief       Serial port code. Supports up to 6 serial ports on the card.
 *
 * \author      Andrew Salmon (copied from serialport.c in QM-RX code)
 *
 * \date        09/22/2008
 *
 */

#include "Cresnet.h"
#include "cresnet_slave.h"
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include "dm_os.h"
#include "memorymanager.h"
#include "serialport.h"
#include "hardware.h"
#include "DeviceJoinInterface.h"
#include "errors.h"
#include "dm_console.h"
#include "string_utils.h"
#include "dm_field_debug.h"
#include "sharedtask.h"
#include "dm_device.h"

// Pointer to array of serial port objects and count
SerialPort **g_pSerialPort;
UINT8 NumSerialPorts;
UINT32 g_pSharedTask;
UINT32 g_ulTaskTCB;
/**
 * \author    Andrew Salmon
 * \brief     Callback function for serial threads
 * \date      08/15/2013
 * \param     ArgList - pointer to the serial argument list used 
 *            by shared task
 * \return    INT32 
 * \retval    time to sleep
 **/
INT32 SerialPortThreadCallback(void *ArgList)
{
    SERIAL_PORT_THREAD_ARGUMENTS *pCallbackArg = (SERIAL_PORT_THREAD_ARGUMENTS*)ArgList;

	SerialPort * pSerialPort = GetSerialPort(pCallbackArg->Channel);
	if (pSerialPort)
    {    
        switch(pCallbackArg->eThreadType)
        {
        case SERIAL_PORT_RX_THREAD_TYPE:
            return pSerialPort->SerialRXThread();
        case SERIAL_PORT_TX_THREAD_TYPE:
            return pSerialPort->SerialTXThread();
        default:
            break;
        }
    }
    DmSystemError(DM_ERROR_LEVEL_FATAL, DM_ERROR_SUBSYS_SERIAL, ERR_SERIAL_INVALID_THREAD_ARG_TYPE, 0);
    return SERIALPORT_NULLPTR_RECALL_DELAY;
}
/**
 * \author    Andrew Salmon
 * \brief     Callback function for serial transmit main task;  This is a C wrapper
 *            to access the non-static method of the serial port class; once called,
 *            the method never returns
 * \date      07/21/2009
 * \param     param - pointer to the sharedTask object
 * \return    void
 * \retval    none
 **/
static void SerialPortTaskCallback(UINT32 param)
{
    SharedTask * pSharedTask = (SharedTask * )param;
    if(pSharedTask)
    {
       pSharedTask->Run();
    }
    else
    {
        DmSystemError(DM_ERROR_LEVEL_FATAL, DM_ERROR_SUBSYS_SERIAL, ERR_SERIAL_INVALID_THREAD_ARG_TYPE, 1);
    }
}
/**
 * \author      Andrew Salmon
 *
 * \date        08/15/2013
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Setup single shared serial port task 
 * \param       bNumSp - number of serial ports
 * \param       pSerialCfg - base serial port configuration 
 * \param       pName - Task name 
 * \param       TaskPrio - task priority
 * \param       stacksize - task stack size 
 * \param       pTaskTCB - Save TCB pointer
 */
SharedTask * SetupSerialPortTask(UINT8 bNumSp, const SerialPortCfg *pSerialCfg, const char *pName, UINT8 TaskPrio, UINT32 stacksize, UINT32 *pTaskTCB)
{
    SHARED_SUBTASK_ENTRY *pSharedtaskList, *pSharedtaskListTmp;
    SERIAL_PORT_THREAD_ARGUMENTS *pThreadArgList, *pThreadArgListTmp;
    const SerialPortCfg *pSerialCfgTmp = pSerialCfg;
    
    // (bNumSp * SERIAL_PORT_MAX_THREAD_TYPE) = 2 = One entry for Tx thread and one for Rx thread, per instance
    pThreadArgList     = (SERIAL_PORT_THREAD_ARGUMENTS*) OsAllocMemory(bNumSp * SERIAL_PORT_MAX_THREAD_TYPE * sizeof(SERIAL_PORT_THREAD_ARGUMENTS));
    pSharedtaskList = (SHARED_SUBTASK_ENTRY*) OsAllocMemory(bNumSp * SERIAL_PORT_MAX_THREAD_TYPE * sizeof(SHARED_SUBTASK_ENTRY));

    if((pThreadArgList == NULL) || (pSharedtaskList == NULL))
    {
        DmSystemError(DM_ERROR_LEVEL_FATAL, DM_ERROR_SUBSYS_SERIAL, ERR_SERIAL_NO_MEMORY, 2);
    }
    
    pThreadArgListTmp     = pThreadArgList;
    pSharedtaskListTmp    = pSharedtaskList;

    for (int iSp = 0 ; iSp < bNumSp; iSp++)
    {
        for(int iThread=SERIAL_PORT_RX_THREAD_TYPE; iThread < SERIAL_PORT_MAX_THREAD_TYPE; iThread++)
        {
            // Populate the serialport thread argument list
            pThreadArgListTmp->eThreadType    = (SERIAL_PORT_THREAD_TYPE)iThread;
            pThreadArgListTmp->Channel        = pSerialCfgTmp->Channel;

            // Setup sharedrtask parameters
            pSharedtaskListTmp->SubTaskPtr    = SerialPortThreadCallback;
            pSharedtaskListTmp->argList       = (void *)&pThreadArgListTmp->eThreadType;

            pThreadArgListTmp++;
            pSharedtaskListTmp++;
        }
        
        pSerialCfgTmp++;
    }

    SharedTask * sharedTask = new SharedTask(pSharedtaskList, bNumSp*SERIAL_PORT_MAX_THREAD_TYPE, (UINT32) 0/*dummy*/);
    // use a single task for all ports
    *pTaskTCB = OsCreateNamedAdvTask(SerialPortTaskCallback, 
                    stacksize, 
                    (UINT32*)sharedTask,
                    TaskPrio,
                    (const signed char *) pName);

    return sharedTask;
    
}
/**
 * \author      Andrew Salmon
 *
 * \date        08/15/2013
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       destroy shared serial port task 
 * \param       ulSharedTask - SharedTask handle
 * \param       ulTaskTCB - task TCB
 *
 */
void DestroySerialPortTask(UINT32 ulSharedTask, UINT32 ulTaskTCB)
{
    SHARED_SUBTASK_ENTRY *pSharedtaskList;
    SERIAL_PORT_THREAD_ARGUMENTS *pThreadArgList;
    SharedTask * pSharedTask = (SharedTask *)ulSharedTask;

    if(pSharedTask)
    {
        pSharedtaskList = pSharedTask->GetEntryPoint();
        pThreadArgList  = (SERIAL_PORT_THREAD_ARGUMENTS *)pSharedtaskList->argList;
    
        //delete task instance
        if ( ulTaskTCB )
        {
            OsSuspendAdvTask(ulTaskTCB);
            OsDeleteAdvTask(ulTaskTCB);
        }

        (void)OsFreeMemory((void*)pSharedtaskList);
        (void)OsFreeMemory((void*)pThreadArgList);

        delete pSharedTask;
    }
    return;
}
/**
 * \author      Andrew Salmon
 *
 * \date        07/23/2009
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Initialize the serial ports
 * \param     bNumSp - number of serial ports
 * \param     pSerialCfg - base serial port configuration 
 * \param      bUseSingleSharedTask - use single task for all 
 *             serialports in system
 */
void CreateSerialPorts(UINT8 bNumSp, const SerialPortCfg *pSerialCfg, BOOL bUseSingleSharedTask)
{
    SerialPort **pCurrentSerialPort;    
    SerialPort *pTmpSerialPort;
    char TaskName[8];
    const SerialPortCfg *pSerialCfgTmp = pSerialCfg;
    g_pSharedTask = 0;
    g_ulTaskTCB   = 0;

    // remember how many serialports
    NumSerialPorts = bNumSp;

    // create an array of pointers to the serial ports
    g_pSerialPort = (SerialPort **) OsAllocMemory(bNumSp * sizeof(SerialPort *));
    DmFatalOnZeroPointer((void*)g_pSerialPort, DM_ERROR_SUBSYS_SERIAL, ERR_SERIAL_NO_MEMORY, 0);

    pCurrentSerialPort = g_pSerialPort;

    // for each serial port instance
    for (int iSp = 0; iSp < bNumSp; iSp++)
    {
        // Stack size of 0 will set it to default size
        UINT16 TaskStackSize = (pSerialCfgTmp->TaskStackSize == 0) ? SERIAL_TASK_STACK_SIZE : pSerialCfgTmp->TaskStackSize;

        // create a driver with the appropriate configuration for each of the transmitters.
        *pCurrentSerialPort = new SerialPort(pSerialCfgTmp);
        pTmpSerialPort = (SerialPort*) *pCurrentSerialPort;

		if (!bUseSingleSharedTask)
		{
            // create the task for each serialport
            sprintf((char *) &TaskName[0], "SERIAL%d", iSp);
            pTmpSerialPort->m_pSharedTask = (UINT32)SetupSerialPortTask(1 , pSerialCfgTmp, &TaskName[0], pSerialCfgTmp->taskPrio, TaskStackSize, &pTmpSerialPort->m_TaskTCB);
		}
        
        // get next serial pointers
        pCurrentSerialPort++;
        pSerialCfgTmp++;
    }

    // if single serial port task for system
    if(bUseSingleSharedTask)
    {
        g_pSharedTask = (UINT32)SetupSerialPortTask(bNumSp, pSerialCfg, "SERIAL", SERIAL_TASK_PRIORITY, SERIAL_TASK_STACK_SIZE, &g_ulTaskTCB);
    }
    return;
}

/**
 * \author      Andrew Salmon
 *
 * \date        08/19/2013
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       destroy Serial objects
 */
void DestroySerialPorts(void)
{
    SerialPort **pCurrentSerialPort;
    SerialPort *pTmpSerialPort;    

    pCurrentSerialPort = g_pSerialPort;

    // for each serial port instance
    for (int iSp = 0; iSp < NumSerialPorts; iSp++)
    {
        pTmpSerialPort = (SerialPort*) *pCurrentSerialPort;

        if(pTmpSerialPort)
        {
            // shutdown serial driver
            pTmpSerialPort->Reconfigure(false);
    
            if (pTmpSerialPort->m_pSharedTask)
            {
                DestroySerialPortTask(pTmpSerialPort->m_pSharedTask , pTmpSerialPort->m_TaskTCB);
            }
    
            // delete drivers
            delete pTmpSerialPort;
            pTmpSerialPort = NULL;
        }
        
        // get next serial pointers
        pCurrentSerialPort++;
    }

    // if single serial port task for system
    if(g_pSharedTask)
    {
        DestroySerialPortTask(g_pSharedTask, g_ulTaskTCB);
    }

    NumSerialPorts = 0;
    (void)OsFreeMemory((void*)g_pSerialPort);
    g_pSerialPort = 0;
    return;
}
/**
 * \author        Andrew Salmon
 * \brief         Distructor for serial port class
 * \date          08/19/2013
 * \return        void
 * \retval        none
 **/
SerialPort::~SerialPort()
{
    if(m_pSerialCfg->pTxBuf == NULL) // memory was dynamicaly allocated
    {
        if(m_pSerialCfg->IsDynamicUartConfig)
        {
            // caller must set m_pTxBuf = NULL, if successfully freed
            DeInitTxBufferPointer();
        }
        if(m_pTxBuf != NULL)
        {
            (void)OsFreeMemory((void*)m_pTxBuf);
            m_pTxBuf = NULL;
        }
    }

    if(m_pSerialCfg->pRxBuf == NULL) // memory was dynamicaly allocated
    {
        if(m_pSerialCfg->IsDynamicUartConfig)
        {
            // caller must set m_pRxBuf = NULL, if successfully freed
            DeInitRxBufferPointer();
        }
        if(m_pRxBuf != NULL)
        {
            (void)OsFreeMemory((void*)m_pRxBuf);
            m_pRxBuf = NULL;
        }
    }

    // Destroy message queue if external UART exist
    DestroyExternalUartMsgQueue();
}
/**
 * \author        Andrew Salmon
 * \brief         constructor for serial port class
 * \date          07/21/2009
 * \param         pSerialCfg -pointer to the configuration for this serial port
 *
 * \return        void
 * \retval        none
 **/
SerialPort::SerialPort(const SerialPortCfg *pSerialCfg) :
    m_iSettleTimer(DM_LIBRARY_DEBOUNCE_DISABLED),
    m_TaskTCB(0),
    m_ChangingComSpec(NEW_COMSPEC_NONE),
    m_OverFlowFlag(0),
    m_CurrentCTS(SERIAL_DATA_INIT_STATE),
    m_EnableReportCTS(0),
    m_StoppedTX(0),
    m_StoppedRX(0),
    m_HandshakeCTS(0),
    m_HandshakeRTS(0),
    m_HandshakeXOFFT(0),
    m_HandshakeXOFFR(0),
    m_CurrentXOFFT(0),
    m_PaceTX(0),
    m_ActiveTX(0),
    m_Stream(0),
    m_UseTxCompInt(0),
    m_pTxBuf(NULL),
    m_pRxBuf(NULL),
    m_BaudRate(0),
    m_PaceCount(0),
    m_stopbits(0),
    m_parity(0),
    m_wlenght(0),
    m_Protocol(0),
    m_Txin(0),
    m_Txout(0),
    m_Rxin(0),
    m_Rxout(0),
    m_pSharedTask(0)
{
    DmFatalOnZeroPointer((void*)pSerialCfg, DM_ERROR_SUBSYS_SERIAL, ERR_SERIAL_NO_CONFIG_TABLE, 0);

    m_pSerialCfg = pSerialCfg;

    //the buffers can be allocated statically by the project of dynamically by the driver
    //if the pointer are null we will assume that the project did not allocate any
    //This was done to minimize change in existing project that had statica allocation
    ////////////////////////////////////////////////////////////////////////////////////aws

    //assign the tx/rx buffers
    m_pTxBuf = m_pSerialCfg->pTxBuf;
    m_pRxBuf = m_pSerialCfg->pRxBuf;

    //check buffers
    if (m_pTxBuf == NULL)
    {
        if(m_pSerialCfg->IsDynamicUartConfig)
        {
            m_pTxBuf = InitTxBufferPointer();
        }

        if (m_pTxBuf == NULL)
        {
            //allocate dynamically
            m_pTxBuf = (UINT8 *) OsAllocMemory(m_pSerialCfg->TxRxBufferSize);
        }
    }

    //check buffers
    if (m_pRxBuf == NULL)
    {
        if(m_pSerialCfg->IsDynamicUartConfig)
        {
            m_pRxBuf = InitRxBufferPointer();
        }

        if (m_pRxBuf == NULL)
        {
            //allocate dynamically
            m_pRxBuf = (UINT8 *) OsAllocMemory(m_pSerialCfg->TxRxBufferSize);
        }
    }
    //////////////////////////////////////////////////////////////////////////////////////////
    if (!m_pTxBuf || !m_pRxBuf)
    {
       DmSystemError(DM_ERROR_LEVEL_FATAL, DM_ERROR_SUBSYS_SERIAL, ERR_SERIAL_NO_MEMORY, 1);
    }

    memset(m_NewComSpec, 0, 4);
    //set default comspec
    //9600, N,8,1
    m_NewComSpec[0] = (UINT8) COMSPEC_9600 | COMSPEC_8BITS;

    // Set driver to use TX complete interrupt for sending data in addition to TX empty
    m_UseTxCompInt = m_pSerialCfg->UseTxCompInt;

    //init join state: multiple serial ports are subslotted and that's reflected in the join number
    GetState(m_Stream)->CresnetSerialMemorize(m_pSerialCfg->SerialOutputJoin, GetNetDestination());

    // Create message queue if external UART exist
    CreateExternalUartMsgQueue();

    // do dynamic initialization
    if(pSerialCfg->IsDynamicUartConfig)
    {
        // start up off until uart detect is complete
        Reconfigure(false);
    }
    else
        Reconfigure(true);
}
/**
 * \author    Andrew Salmon
 * \brief     Monitors serial tx buffer and send a character on each serial port from tx queue
 * \date      10/02/2008
 * \param     param (not used)
 * \return    milliseconds to sleep
 * \retval    none
 **/
INT32 SerialPort::SerialTXThread(void)
{
    //service external uart if it exist
    ServiceExternalUart();

    if (m_ChangingComSpec != NEW_COMSPEC_NONE)
    {
        if (!m_pSerialCfg->ExternalUART)
            return (SERIAL_TASK_PERIOD);
    }

    if (m_EnableReportCTS)
    {
      UINT8 state = !GetCTS(); // Read cts pin status
        if (state != m_CurrentCTS)
        {
            m_CurrentCTS = state;
         DeviceSendRawDigital( m_pSerialCfg->CTS_DigitalOutputJoin, m_CurrentCTS, m_Stream, true );
        }
    }

    // Always read CTS before loading next character
    m_StoppedTX = ( m_HandshakeCTS && GetCTS() ) || ( m_HandshakeXOFFT && m_CurrentXOFFT );

    if (!m_ActiveTX && !m_StoppedTX)
    {
#ifndef SERIES_3_CONTROL
        TxLoadNextChar();
#else
        // For half duplex, just start up the TX interrupt and let the ISR handle the entire TX
        if (m_Protocol != COMSPEC_PROTOCOL_RS485)
        TxLoadNextChar();
        else
        EnableTxInt();
#endif
    }

    if (m_pSerialCfg->ExternalUART)
        return 0; //can not sleep need to service external device

    //if pacing supported..use that value
    if (m_PaceTX)
    {
        // sleep for a while
        return (m_PaceCount);
    }
    else
    {
        // sleep for a while
        return (SERIAL_TASK_PERIOD);
    }
}
/**
 * \author    Andrwe Salmon
 * \brief     Monitors serial rx buffer for data received from the external serial device
 * \date      10/02/2008
 * \param     param (not used)
 * \return    ms to sleep
 * \retval    none
 **/
INT32 SerialPort::SerialRXThread(void)
{
    UINT8 *ptr, *string, Xdata;
    UINT32 tmpCount, i;

    if ((m_Rxin == m_Rxout) || (m_ChangingComSpec == WRITE_NEW_COMSPEC))
    {
        if (m_ChangingComSpec == WRITE_NEW_COMSPEC)
        {
            m_ChangingComSpec = WRITING_NEW_COMSPEC;
            SetNewComSpec();

            if (m_ChangingComSpec == WRITING_NEW_COMSPEC)
                m_ChangingComSpec = NEW_COMSPEC_NONE;

            return 0; // yield
        }
        //if rx flow blocked and now below midway mark
        if (m_StoppedRX)
        {
            if (m_HandshakeRTS)
            {
                AssertRTS(false);
                m_StoppedRX = false;
            }

            if (m_HandshakeXOFFR)
            {
                Xdata = XON;
                //send this flow control char immediately
                TxSerialData(&Xdata, 1, m_pSerialCfg->sendParam);
                m_StoppedRX = false;
            }

        }
        // sleep for a while
        return (SERIAL_TASK_PERIOD);
    }

    //Check if we need the feedback settling timer
    if (m_pSerialCfg->FeedbackSettleTime > 0)
    {
        //Check if we have data to send
        tmpCount = GetRxCount();
        if (tmpCount > 0)
        {
            //Check if we should short circuit the delay because we're falling too far behind
            if ( tmpCount >= m_pSerialCfg->TxRxReleaseHighWaterMark/2 && m_iSettleTimer > 0 )
            {
                if (IsDmFieldDebugIndexActive(DM_FIELD_DM_SERIAL_PORT_DEBUG_IDX))
                {
                    DmDebugPrintf("SerialPort - Skip settle time\r\n");
                }
                m_iSettleTimer = 0;
            }

            //Check if timer is currently disabled
            if (m_iSettleTimer == DM_LIBRARY_DEBOUNCE_DISABLED)
            {
                //Start the timer from the beginning
                m_iSettleTimer = m_pSerialCfg->FeedbackSettleTime;
            }

            //Check if we're waiting for the data to settle
            if (m_iSettleTimer > 0)
            {
                //Decrement the timer and wait until it has elapsed
                m_iSettleTimer--;
                return (SERIAL_TASK_RX_SETTLE_POLL_MS);
            }
        }
    }

    if (( !IsCrestnetSlaveQueueFull(m_Stream, GetNetDestination()) )||( IsDmFieldDebugIndexActive(DM_FIELD_DM_SERIAL_PORT_DEBUG_IDX) ))
    {
        if (!MemMgr || (string = MemMgr->GetBlock(MAX_SERIAL_PACKET)) == 0)
            return 0; // yield

        ptr = string;
        tmpCount = GetRxCount();

        if (tmpCount <= MAX_SERIAL_PACKET)
        {
            //Disable the timer once we know the queue will be emptied
            m_iSettleTimer = DM_LIBRARY_DEBOUNCE_DISABLED;
        }

        //how many to send
        tmpCount = min( tmpCount, MAX_SERIAL_PACKET );

        for (i = 0; i < tmpCount; i++)
        {
            // bug 52689: when IAR is set to high optimization.only this simplified version works. making variables volatile is not an option
            *ptr = m_pRxBuf[m_Rxout];
            ptr++;
            m_Rxout++;
            /// bug 52689

            if (m_Rxout >= m_pSerialCfg->TxRxBufferSize)
                m_Rxout = 0;
        }

        if (!IsCrestnetSlaveQueueFull(m_Stream, GetNetDestination()))
        {
            //flag this event with a NULL string
            GetState(m_Stream)->CresnetSerialMemorize(m_pSerialCfg->SerialOutputJoin, GetNetDestination());
            CresnetSlaveSendSerial(m_Stream, string, tmpCount, m_pSerialCfg->SerialOutputJoin, GetNetDestination());
        }
        else
        {
            DmSystemError(DM_ERROR_LEVEL_ERROR,DM_ERROR_SUBSYS_SERIAL,ERR_SERIAL_CRESTNET_SLAVE_QUEUE_FULL,(UINT16)m_pSerialCfg->Channel);
        }

        if (IsDmFieldDebugIndexActive(DM_FIELD_DM_SERIAL_PORT_DEBUG_IDX))
        {
            // Debug message
            DmDebugPrintf("COM%1d RX: [", m_pSerialCfg->Channel);

            ptr = string;
            for (i = 0; i < tmpCount; i++)
            {
                DmDebugPrintf(" %02x", *ptr++);
            }

            DmDebugPrintf(" ]\r\n");
        }

        MemMgr->FreeBlock(string);

        //if rx flow blocked and now below midway mark
        if (m_StoppedRX)
        {
            //check Rxin lead
            if (m_Rxin >= m_Rxout)
            {
                tmpCount = m_Rxin - m_Rxout;
            }
            else
            {
                //buffer wrapped
                tmpCount = m_Rxin + ( m_pSerialCfg->TxRxBufferSize - m_Rxout );
            }

            //if lead is less that 60% start receiving again
            if (tmpCount < m_pSerialCfg->TxRxReleaseHighWaterMark)
            {
                if (m_HandshakeRTS)
                {
                    AssertRTS(false);
                    m_StoppedRX = false;
                }

                if (m_HandshakeXOFFR)
                {
                    Xdata = XON;
                    //send this flow control char immediately
                    TxSerialData(&Xdata, 1, m_pSerialCfg->sendParam);
                    m_StoppedRX = false;
                }
            }
        }
    }

    return 0; // yield
}

/**
 * \author      Adolfo Velasco
 * \date        03/20/2013
 * \brief       Gets # of Rx bytes
 * \return      none
 * \retval      void
 * \param
 */
UINT32 SerialPort::GetRxCount(void)
{
    //Use temporary variables because we are not semaphore protected
    UINT32 tmpIn = m_Rxin;
    UINT32 tmpOut = m_Rxout;

    if (tmpIn == tmpOut)
    {
        return 0;
    }
    else if (tmpIn > tmpOut)
    {
        return (tmpIn - tmpOut);
    }
    else
    {
        return (tmpIn + (m_pSerialCfg->TxRxBufferSize - tmpOut));
    }
}

/**
 * \author      Andrew Salmon
 * \date        09/19/2008
 * \brief       Copy cresnet data to serial port tx buffer
 * \return      none
 * \retval      void
 * \param       ptr - pointer to the rx data
 * \param       count - number of data bytes
 */
void SerialPort::ReceiveTxData(char *ptr, UINT32 count)
{
    // assert enabled
    if (!m_bTxEnable)
        return;

    // Debug message
    if (IsDmFieldDebugIndexActive(DM_FIELD_DM_SERIAL_PORT_DEBUG_IDX))
    {
        DmDebugPrintf("COM%1d TX: [", m_pSerialCfg->Channel);

        UINT8* pTemp = (UINT8*) ptr;

        for (UINT32 i = 0; i < count; i++)
        {
            DmDebugPrintf(" %02x", *pTemp++);
        }

        DmDebugPrintf(" ]\r\n");
    }

    while (count--)
    {
        OsEnterCritical();
        m_pTxBuf[m_Txin++] = *ptr++;

        //wrap circular buffer
        if (m_Txin >= m_pSerialCfg->TxRxBufferSize)
            m_Txin = 0;

        OsExitCritical();
    }
    return;
}
/**
 * \author      Andrew Salmon
 * \date        09/19/2008
 * \brief       Returns pointer to serial port instance
 * \return      pointer to serial port
 * \retval      SerialPort
 * \param       Channel - channel number (0-based, channel A = 
 *              0)
 *
 */
SerialPort *GetSerialPort(UINT8 Channel)
{
    SerialPort *pTmpSerialPort;

    if(g_pSerialPort)
    {
        for(int i=0; i<NumSerialPorts; i++)
        {
            pTmpSerialPort = *(g_pSerialPort + i);
            if(pTmpSerialPort && (pTmpSerialPort->GetChannel() == Channel))
            {
                return pTmpSerialPort;
            }
        }
    }
    return NULL;
}
/**
 * \author      Andrew Salmon
 * \date        09/19/2008
 * \brief       Copy cresnet data to serial port tx buffer
 * \return      none
 * \retval      void
 * \param       bStream - channel number (0-based, channel A = 0)
 * \param       ptr - pointer to the rx data
 * \param       count - number of data bytes
 */
void SerialPortReceiveTxData(UINT8 Channel, char *ptr, UINT32 count)
{
    SerialPort *pTmpSerialPort;

    pTmpSerialPort = GetSerialPort(Channel);

    if (pTmpSerialPort)
        pTmpSerialPort->ReceiveTxData(ptr, count);

    return;
}
/**
 * \author      Andrew Salmon
 * \date        09/18/2008
 * \brief       Send a character on each serial port from tx queue
 * \return      none
 * \retval      void
 * \param       pCurrent - pointer to the serial port data
 */
void SerialPort::TxLoadNextChar(void)
{
    UINT32 tmpOut;

    if (m_Txin == m_Txout)
        return;

    if (!m_PaceTX)
    {
        OsEnterCritical();

        tmpOut = m_Txout++;
        //wrap circular buffer
        if (m_Txout >= m_pSerialCfg->TxRxBufferSize)
            m_Txout = 0;

        m_ActiveTX = true;
        //do all the stuff above before enabling the tx interrupt
        TxSerialDataINT(m_pSerialCfg->sendParam, m_pTxBuf[tmpOut]);

        OsExitCritical();
    }
    else
    {
        tmpOut = m_Txout++;
        //wrap circular buffer
        if (m_Txout >= m_pSerialCfg->TxRxBufferSize)
            m_Txout = 0;

      TxSerialData((unsigned char *)&m_pTxBuf[tmpOut], 1, m_pSerialCfg->sendParam);
    }

}
/**
 * \author      Andrew Salmon
 *
 * \date        09/22/2008
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       (Upper-level) Interrupt service routine for
 *              the UART console.
 *
 * \param       Channel - COMM port channel
 * \param       gotBreak : 1 if we received a "break", 0 otherwise
 * \param       rxReady : 1 if we received at least 1 character
 * \param       newByte : the character just received
 * \param       txEmpty : 1 if the transmitter is done sending something
 * \warning     Runs from interrupt context.
 *
 */
UINT32 SerialPortIsr(UINT32 Channel, BOOL gotBreak, BOOL rxReady, UINT8 newByte, BOOL txEmpty)
{
    SerialPort *pTmpSerialPort;

    pTmpSerialPort = GetSerialPort(Channel);

    if (pTmpSerialPort)
    {
      return (pTmpSerialPort->TxRxIsr(gotBreak, rxReady, newByte, txEmpty));
    }

    return OS_ISR_FLAG_NO_MORE_DATA;
}
/**
 * \author      Andrew Salmon
 *
 * \date        09/22/2008
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       (Upper-level) Interrupt service routine for
 *              the UART console.
 *
 * \param       gotBreak : 1 if we received a "break", 0 otherwise
 * \param       rxReady : 1 if we received at least 1 character
 * \param       newByte : the character just received
 * \param       txEmpty : 1 if the transmitter is done sending something
 * \warning     Runs from interrupt context.
 *
 */
UINT32 SerialPort::TxRxIsr(BOOL gotBreak, BOOL rxReady, UINT8 newByte, BOOL txEmpty)
{
    UINT8 Xdata;
    UINT32 tmpCount, isrFlags = 0;

    if( m_ChangingComSpec != NEW_COMSPEC_NONE )
        return OS_ISR_FLAG_NO_MORE_DATA;

    if (rxReady)
    {
        if ((m_HandshakeXOFFT) && ((newByte == XOFF) || (newByte == XON)))
        {
            if (newByte == XOFF)
                m_CurrentXOFFT = true;
            else//if ( newByte == XON )
                m_CurrentXOFFT = false;
        }
        else
        {
            if (!m_StoppedRX)
            {
                // bug 52689: when IAR is set to high optimization.only this simplified version works. making variables volatile is not an option
                m_pRxBuf[m_Rxin] = newByte;
                m_Rxin++;
                // bug 52689

                //wrap circular buffer
                if (m_Rxin >= m_pSerialCfg->TxRxBufferSize)
                    m_Rxin = 0;
            }
        }

        if (!m_StoppedRX)
        {
            //check Rxin lead
            if (m_Rxin >= m_Rxout)
            {
                tmpCount = m_Rxin - m_Rxout;
            }
            else
            {
                //buffer wrapped
                tmpCount = m_Rxin + (m_pSerialCfg->TxRxBufferSize - m_Rxout);
            }

            if (tmpCount > m_pSerialCfg->TxRxHighWaterMark)
            {
                if (!m_OverFlowFlag)
                {
                    //send error message serial buffer overrflow
                 DmSystemError(DM_ERROR_LEVEL_ERROR,DM_ERROR_SUBSYS_SERIAL,ERR_SERIAL_RX_BUFFER_OVERFLOW,(UINT16)m_pSerialCfg->Channel);
                    m_OverFlowFlag = true;
                    ReportOverflow();
                }
                //try to stop serial divice from sending
                if (m_HandshakeRTS)
                {
                    AssertRTS(true);
                    m_StoppedRX = true;
                }
                if (m_HandshakeXOFFR)
                {
                    Xdata = XOFF;
                    //send this flow control char immediately
                    TxSerialData(&Xdata, 1, m_pSerialCfg->sendParam);
                    m_StoppedRX = true;
                }
            }
        }
    }
    if (txEmpty)
    {
        if (m_ActiveTX)
        {
          m_StoppedTX = ( m_HandshakeCTS && GetCTS() ) || ( m_HandshakeXOFFT && m_CurrentXOFFT );

            if (!m_StoppedTX)
            {
                if (m_Txin == m_Txout)
                {
                    m_ActiveTX = false;
                    isrFlags = OS_ISR_FLAG_NO_MORE_DATA;

                    // RS485 is half duplex only
                    if (m_Protocol == COMSPEC_PROTOCOL_RS485)
                        EnableReceive(); // disable TX and switch the bus
                }
                else
                {
                    //tx the next character
                    isrFlags |= m_pTxBuf[m_Txout++];

                    //wrap circular buffer
                    if (m_Txout >= m_pSerialCfg->TxRxBufferSize)
                        m_Txout = 0;

                    // If last byte of the transfer and half duplex mode (RS485 only)
                if ((m_Protocol == COMSPEC_PROTOCOL_RS485) && (m_Txin == m_Txout))
                    {
                        // enable Transfer complete interrupt instead of Tx Empty
                        EnableTransferComplete();
                    }
                }
            } // if ( !m_StoppedTX  )
            else
            {
                m_ActiveTX = false;
                isrFlags = OS_ISR_FLAG_NO_MORE_DATA;
            }
        }
        else
        {
            isrFlags = OS_ISR_FLAG_NO_MORE_DATA;
        }

        return isrFlags;        // done
    }

    // Some devices use TC in addition to instead of just TXE exclusively thus we don't have to tell the task
    // to start another transfer; TX data will be loaded via uart ISR
    else if (m_ActiveTX && !m_UseTxCompInt)
    {
        //Tx interrupt will be disabled by caller (rx interrupt)
        //So kick start tx from the task..
        m_ActiveTX = false;
    }

    // nothing to tx - disable int
    return OS_ISR_FLAG_NO_MORE_DATA;
}

/**
 * \author      Andrew Salmon
 * \date        09/16/2008
 * \brief       Serial port specs as received from control system
 * \return      none
 * \retval      void
 * \param       char pointer to the cresnet packet
 */
void SerialPortReceiveComspec(UINT8 *ptr, UINT8 Channel)
{
    SerialPort *pTmpSerialPort;

    if (*(ptr + 1) == 0)
    {   
        pTmpSerialPort = GetSerialPort(Channel); //get the channel from the 3 ls bit

        if (pTmpSerialPort)
            pTmpSerialPort->ReceiveComspec(ptr);
    }
    return;
}
/**
 * \author      Andrew Salmon
 * \date        09/16/2008
 * \brief       Serial port specs as received from control system
 * \return      none
 * \retval      void
 * \param       char pointer to the cresnet packet
 */
void SerialPort::ReceiveComspec(UINT8 *ptr)
{
    if( ( m_pTxBuf == NULL )||( m_pRxBuf == NULL ) || ( m_pSerialCfg->TxRxBufferSize == 0) )
    {
        //send error message serial buffer overrflow
        DmSystemError(DM_ERROR_LEVEL_WARNING,DM_ERROR_SUBSYS_SERIAL,ERR_SERIAL_BUFFER_OVERFLOW_NULL_PTR,(UINT16)m_pSerialCfg->Channel);
        return;
    }

    memcpy(m_NewComSpec, ptr + 2, 4); //4 bytes comspec_low, comspec_hi, pacing, max_len
    //  <id> <cnt> <12> <80h+chan> <00> <comspec_low> <comspec_hi> <pacing> <max len>
    //    <comspec_low>, <comspec_hi> are build with the below table.
    //    <pacing>:  in milliseconds (31 ms max)
    //   <max len>: max string length, 246 max, 0 means use default

    m_ChangingComSpec = WRITE_NEW_COMSPEC;

#ifdef SERIES_3_CONTROL
    // Makes this function block to prevent data from being sent ahead of comspec being set on multiple ports
    UINT8 Retries = 5;

    // Wait until the comspec is set or retries is up (1 retry should be enough for the comspec to be set)
    while ((m_ChangingComSpec == WRITE_NEW_COMSPEC) && (Retries-- > 0))
    HwDelayMsec(SERIAL_TASK_PERIOD);
#endif

    return;
}
/**
 * \author    Andrew Salmon - partially ported from AMS-AIP
 * \brief     Save new com port parameters to class and configure hardware
 * \date      04/17/2009
 * \param       bps - baud rate
 * \param       stopBits - number of stop bits
 * \param       parity - aprity
 * \param       wlenght - word length
 * \param       hwcontrol - hardware flow control
 **/
void SerialPort::SetNewComParameters(UINT32 bps, UINT32 stopBits, UINT32 parity, UINT32 wlenght, UINT32 hwcontrol)
{
    m_Txin = 0;
    m_Rxin = 0;
    m_Rxout = 0;
    m_Txout = 0;
    m_ActiveTX = false;
    m_BaudRate = bps;
    m_stopbits = stopBits;
    m_parity = parity;
    m_wlenght = wlenght;
    m_StoppedTX = 0;
    m_StoppedRX = 0;
    m_CurrentXOFFT = 0;

   UartConfig(m_pSerialCfg->sendParam, bps, stopBits, parity, wlenght,hwcontrol);
}
/**
 * \author    Andrew Salmon
 * \brief     set hardware handshake
 * \date      07/21/2009
 * \param       enable - enable/disable handshake
 *
 **/
void SerialPort::EnableHwHandShake(BOOL enable)
{
    if (enable)
    {
        m_HandshakeCTS = true;
        m_HandshakeRTS = true;

        m_NewComSpec[1] |= (UINT8) ((COMSPEC_CTS | COMSPEC_RTS) >> 8);
        AssertRTS(false);
    }
    else
    {
        m_HandshakeCTS = false;
        m_HandshakeRTS = false;
        m_NewComSpec[1] &= ~((UINT8) ((COMSPEC_CTS | COMSPEC_RTS) >> 8));
#ifdef SERIES_3_CONTROL
        // Turn off RTS if we're not using it
        AssertRTS(true);
#endif
    }
}

/**
 * \author    Andrew Salmon
 * \brief     set hardware handshake
 * \date      07/21/2009
 * \param       enable - enable/disable handshake
 *
 **/
BOOL SerialPort::ReturnHwHandShake(void)
{
    return m_HandshakeRTS;
}
/**
 * \author    Andrew Salmon
 * \brief     returns current baud, stopbits, parity and wordlength
 * \date      07/21/2009
 * \param     pbaud - dest for baudrate
 * \param     pstopbits - dest for stopbits
 * \param     pparity - dest for parity
 * \param     pwordlength - dest for wordlength
 *
 **/
void SerialPort::GetHwConfig(UINT32 *pbaud, UINT32 *pstopbits, UINT32 *pparity, UINT32 *pwordlength)
{
    *pbaud = m_BaudRate;
    *pstopbits = m_stopbits;
    *pparity = m_parity;
    *pwordlength = m_wlenght;
}
/**
 * \author      Andrew Salmon
 *
 * \date        09/18/2008
 *
 * \brief       Change UART RTS state
 *
 * \param       state - Bit_SET (asserts)/Bit_Reset (deasserts)
 */
void SerialPort::SetRTSToDigitalJoinState(BOOL state)
{
    if (!m_HandshakeRTS)
    {
        // assert/deassert RTS
        AssertRTS(state ? false : true);
    }
}

/**
 * \author      Chris Merck
 * \date        2 July 2013
 * \brief       Decode 4-byte comspec into individual values. 
 * \note        Comspec is read-only. All other values modified 
 *  			by reference.
 */
static BOOL DecodeComspec(UINT8 * comspec, 
				   UINT32 & bps, UINT32 & stopbits, UINT32 & parity, UINT32 & wlength, UINT16 & pacing,
				   UINT8 & HandshakeCTS, 
				   UINT8 & HandshakeRTS,
				   UINT8 & HandshakeXOFFT,
				   UINT8 & HandshakeXOFFR,
				   UINT8 & EnableReportCTS)
{
	UINT16 comspec16; 
    comspec16 = MAKEUINT16(*(&comspec[0]),*(&comspec[1])); // first 16-bits

	switch (comspec16 & COMSPEC_BRMASK)
	{
	case COMSPEC_300:
		bps = 300;
		break;

	case COMSPEC_600:
		bps = 600;
		break;

	case COMSPEC_1200:
		bps = 1200;
		break;

	case COMSPEC_1800:
		bps = 1800;
		break;

	case COMSPEC_2400:
		bps = 2400;
		break;

	case COMSPEC_3600:
		bps = 3600;
		break;

	case COMSPEC_4800:
		bps = 4800;
		break;

	case COMSPEC_7200:
		bps = 7200;
		break;

	case COMSPEC_9600:
		bps = 9600;
		break;

	case COMSPEC_14400:
		bps = 14400;
		break;

	case COMSPEC_19200:
		bps = 19200;
		break;

	case COMSPEC_28800:
		bps = 28800;
		break;

	case COMSPEC_38400:
		bps = 38400;
		break;

	case COMSPEC_57600:
		bps = 57600;
		break;

	case COMSPEC_115200:
		bps = 115200;
		break;

	default:
		bps = 38400;
		break;
	}

    pacing = (UINT16) comspec[2];

	switch (comspec16 & COMSPEC_PMASK)
	{
	case COMSPEC_PEVEN:
		parity = UART_CRESNET_PARITY_EVEN;
		break;

	case COMSPEC_PODD:
		parity = UART_CRESNET_PARITY_ODD;
		break;

	case COMSPEC_PNONE:
	default:
		parity = UART_CRESNET_PARITY_NO;
		break;
	}

	switch (comspec16 & COMSPEC_BMASK)
	{
	case COMSPEC_7BITS:
		wlength = 7;
		break;

	case COMSPEC_8BITS:
	default:
		wlength = 8;
		break;
	}

	if ((comspec16 & COMSPEC_SMASK) == COMSPEC_2STOP)
		stopbits = 2;
	else
		stopbits = 1;

	HandshakeCTS = ((comspec16 & COMSPEC_CTS) != 0);
	HandshakeRTS = ((comspec16 & COMSPEC_RTS) != 0);

	HandshakeXOFFT = ((comspec16 & COMSPEC_XOFFT) != 0);
	HandshakeXOFFR = ((comspec16 & COMSPEC_XOFFR) != 0);
	EnableReportCTS = ((comspec16 & COMSPEC_RCTS) != 0);

	return TRUE;
}
static BOOL DecodeComspec(UINT8 * comspec, 
				   UINT32 & bps, UINT32 & stopbits, UINT32 & parity, UINT32 & wlength, UINT16 & pacing)
{
	// don't care about handshaking
	// create dummy vars
	UINT8 HandshakeCTS;
	UINT8 HandshakeRTS;
	UINT8 HandshakeXOFFT;
	UINT8 HandshakeXOFFR;
	UINT8 EnableReportCTS;

	return DecodeComspec(comspec,bps,stopbits,parity,wlength,pacing,
				  HandshakeCTS,HandshakeRTS,HandshakeXOFFT,HandshakeXOFFR,EnableReportCTS);
}

/**
 * \author      Chris Mercj
 * \date        2 July 2013
 * \brief       Encode 4-byte comspec from individual values. 
 * \note        Assumes comspec is 4byte array. All other values
 *  			read-only.
 */
static BOOL EncodeComspec(UINT8 * comspec, 
				   UINT32 bps, UINT32 stopbits, UINT32 parity, UINT32 wlength, UINT16 pacing,
				   UINT8 HandshakeCTS, 
				   UINT8 HandshakeRTS,
				   UINT8 HandshakeXOFFT,
				   UINT8 HandshakeXOFFR,
				   UINT8 EnableReportCTS)
{
	UINT16 comspec16; 
	BOOL success = true;
        
        comspec16 = MAKEUINT16(*(&comspec[0]),*(&comspec[1]));

	switch (bps)
	{
	case 300:
		comspec16 |= COMSPEC_300;
		break;

	case 600:
		comspec16 |= COMSPEC_600;
		break;

	case 1200:
		comspec16 |= COMSPEC_1200;
		break;

	case 1800:
		comspec16 |= COMSPEC_1800;
		break;

	case 2400:
		comspec16 |= COMSPEC_2400;
		break;

	case 3600:
		comspec16 |= COMSPEC_3600;
		break;

	case 4800:
		comspec16 |= COMSPEC_4800;
		break;

	case 7200:
		comspec16 |= COMSPEC_7200;
		break;

	case 9600:
		comspec16 |= COMSPEC_9600;
		break;

	case 14400:
		comspec16 |= COMSPEC_14400;
		break;

	case 19200:
		comspec16 |= COMSPEC_19200;
		break;

	case 28800:
		comspec16 |= COMSPEC_28800;
		break;

	case 38400:
		comspec16 |= COMSPEC_38400;
		break;

	case 57600:
		comspec16 |= COMSPEC_57600;
		break;

	case 115200:
		comspec16 |= COMSPEC_115200;
		break;

	default:
		comspec16 |= COMSPEC_38400;
		success = false;
		break;
	}

	switch (parity)
	{
	case UART_CRESNET_PARITY_EVEN:
		comspec16 |= COMSPEC_PEVEN;
		break;

	case UART_CRESNET_PARITY_ODD:
		comspec16 |= COMSPEC_PODD;
		break;

	default:
		success = false;
	case UART_CRESNET_PARITY_NO:
		comspec16 |= COMSPEC_PNONE;
		break;
	}

	switch (wlength)
	{
	case 7:
		comspec16 |= COMSPEC_7BITS;
		break;

	default:
		success = false;
	case 8:
		comspec16 |= COMSPEC_8BITS;
		break;
	}

	if (stopbits == 2)
	{
		comspec16 |= COMSPEC_2STOP;
	}
	else 
	{	
		if (stopbits != 1)
			success = false;

		comspec16 |= 0; // no flag for 1 stop bit
	}

	if (HandshakeCTS) comspec16 |= COMSPEC_CTS;
	if (HandshakeRTS) comspec16 |= COMSPEC_RTS;

	if (HandshakeXOFFT) comspec16 |= COMSPEC_XOFFT;
	if (HandshakeXOFFR) comspec16 |= COMSPEC_XOFFR;
	if (EnableReportCTS) comspec16 |= COMSPEC_RCTS;

	comspec[0] = LOUINT8(comspec16);
	comspec[1] = HIUINT8(comspec16);
    comspec[2] = pacing;
    comspec[3] = 0;

	return success;
}
static BOOL EncodeComspec(UINT8 * comspec, 
				   UINT32 bps, UINT32 stopbits, UINT32 parity, UINT32 wlength, UINT16 pacing)
{
	// default to no handshaking features
	return EncodeComspec(comspec,bps,stopbits,parity,wlength,pacing,0,0,0,0,0);
}


/**
 * \author      Andrew Salmon
 *
 * \date        01/11/2010
 *
 * \brief       Write new comspec to uart
 *
 */
void SerialPort::SetNewComSpec(void)
{
    UINT32 bps, stopbits, parity, wlenght;
    UINT16 comspec, pacing;

	DecodeComspec(m_NewComSpec,bps,stopbits,parity,wlenght,pacing,
				  m_HandshakeCTS,m_HandshakeRTS,m_HandshakeXOFFT,m_HandshakeXOFFR,m_EnableReportCTS);

    if (pacing > 32)
        pacing = 32;

    m_PaceCount = pacing;
    m_PaceTX = (pacing != 0);

    comspec = MAKEUINT16(*(&m_NewComSpec[0]),*(&m_NewComSpec[1])); // first 16-bits

    SetNewComParameters(bps, stopbits, parity, wlenght, (comspec & (COMSPEC_CTS|COMSPEC_RTS)));
    SetNewProtocol(comspec & COMSPEC_PROTOCOL_MASK);

    if (m_HandshakeRTS)
        AssertRTS(false);

// Should reset RTS if it's not used only in RS232 mode
#ifdef SERIES_3_CONTROL
    else
    {
        if ((comspec & COMSPEC_PROTOCOL_MASK) == COMSPEC_PROTOCOL_RS232)
        AssertRTS(true); // Turn off RTS if HW handshake is not enabled
    }
#endif

    //read cts status
    if (m_EnableReportCTS)
    {
        m_CurrentCTS = !GetCTS();
       DeviceSendRawDigital( m_pSerialCfg->CTS_DigitalOutputJoin, m_CurrentCTS, m_Stream, true );
    }

#ifdef SERIES_3_CONTROL
    ReportComSpec(bps, stopbits, parity, wlenght);

    // Debug prints for 3 series
    if (IsDmFieldDebugIndexActive(DM_FIELD_DM_SERIAL_PORT_DEBUG_IDX))
    {
        DmDebugPrintf("New Comspec Port:%d, bps:%d, stopbits:%d, parity:%d, data:%d, prot:%d, ",
                m_pSerialCfg->Channel, bps, stopbits, parity, wlenght, m_Protocol);

        DmDebugPrintf("CTS:%s, RTS:%s, XONT:%s, XONR:%s, RptCTS:%s\r\n", m_HandshakeCTS ? "On":"Off",
                m_HandshakeRTS ? "On":"Off", m_HandshakeXOFFT ? "On":"Off", m_HandshakeXOFFR ? "On":"Off",
                m_EnableReportCTS ? "On":"Off");
    }
#endif

    return;
}
/**
 * \author      Andrew Salmon
 *
 * \date        01/11/2010
 *
 * \brief       Write new comspec to uart
 *
 */
void SerialPort::SetCommParameters(UINT8 comSpec)
{
    m_NewComSpec[0] = comSpec;
    m_ChangingComSpec = WRITE_NEW_COMSPEC;
}

#ifdef SERIES_3_CONTROL
/**
 *
 * \author      RJen
 * \brief       Reports back the comspec set for the com port to the control system
 * \detail      Reports back the comspec set for the com port to the control system
 * \date        01/1/2013
 * \param       Baudrate - baudrate
 *              StopBits - stop bits
 *              Parity - parity
 *              DataBits - data bits
 * \return      none
 * \retval      none
 * \warning     none
 * \note        Only for 3 series devices
 *
 */
void SerialPort::ReportComSpec(UINT32 Baudrate, UINT8 StopBits, UINT8 Parity, UINT8 DataBits)
{
    CresnetJoinState *pGetState = GetState(m_Stream); // Stream 0
    UINT32 JoinOffset = m_pSerialCfg->Channel * TLDM_COMPORT_JOIN_NUM_OFFSET;// Offset is based on port number

    if (pGetState)
    {
        // Report baudrate, 115.2k baud is too big to fit in an analog value, report back 115
        if (Baudrate == 115200)
        pGetState->CresnetAnalogMemorize(JoinOffset + TLDM_COM_PORT_BAUD_RATE_REPORTING_JOIN, 115, GetNetDestination(), TRUE);
        else
        pGetState->CresnetAnalogMemorize(JoinOffset + TLDM_COM_PORT_BAUD_RATE_REPORTING_JOIN, (UINT16)Baudrate, GetNetDestination(), TRUE);

        // Report data bits, stop bits, parity
        pGetState->CresnetAnalogMemorize(JoinOffset + TLDM_COM_PORT_DATA_BITS_REPORTING_JOIN, (UINT16)DataBits, GetNetDestination(), TRUE);
        pGetState->CresnetAnalogMemorize(JoinOffset + TLDM_COM_PORT_STOP_BITS_REPORTING_JOIN, (UINT16)StopBits, GetNetDestination(), TRUE);

        char ParityValue = 'N';// Default parity: none

        if (Parity == UART_CRESNET_PARITY_EVEN)
        ParityValue = 'E';
        else if (Parity == UART_CRESNET_PARITY_ODD)
        ParityValue = 'O';

        pGetState->CresnetAnalogMemorize(JoinOffset + TLDM_COM_PORT_PARITY_REPORTING_JOIN, (UINT16)ParityValue, GetNetDestination(), TRUE);

        // Report protocol
        UINT8 Protocol = TLDM_COMPORT_SUPPORTS_RS232;// Default protocol: 232

        if (m_Protocol == COMSPEC_PROTOCOL_RS422)
        Protocol = TLDM_COMPORT_SUPPORTS_RS422;
        else if (m_Protocol == COMSPEC_PROTOCOL_RS485)
        Protocol = TLDM_COMPORT_SUPPORTS_RS485;

        pGetState->CresnetAnalogMemorize(JoinOffset + TLDM_COM_PORT_PROTOCOL_REPORTING_JOIN, (UINT16)Protocol, GetNetDestination(), TRUE);

        // Report hardware handshake
        UINT8 HwHandShake = 0;// Default: rts and cts off

        if (m_HandshakeRTS)
        HwHandShake |= TLDM_COMPORT_USE_RTS;

        if (m_HandshakeCTS)
        HwHandShake |= TLDM_COMPORT_USE_CTS;

        pGetState->CresnetAnalogMemorize(JoinOffset + TLDM_COM_PORT_HW_REPORTING_JOIN, (UINT16)HwHandShake, GetNetDestination(), TRUE);

        // Report software handshake
        UINT8 SwHandShake = 0;// Default: xonr xont off

        if (m_HandshakeXOFFT && m_HandshakeXOFFR)
        SwHandShake = TLDM_COMPORT_XONT_XONR_ENABLE;
        else if (m_HandshakeXOFFT)
        SwHandShake = TLDM_COMPORT_XONT_ENABLED;
        else if (m_HandshakeXOFFR)
        SwHandShake = TLDM_COMPORT_XONR_ENABLED;

        pGetState->CresnetAnalogMemorize(JoinOffset + TLDM_COM_PORT_SW_REPORTING_JOIN, (UINT16)SwHandShake, GetNetDestination(), TRUE);
    }
}
#endif // #ifdef SERIES_3_CONTROL

void SerialPort::Reconfigure(BOOL bEnable)
{
    if (bEnable)
    {
        // restore comspec
        m_ChangingComSpec = WRITE_NEW_COMSPEC;

        // reset ring buffers
        m_Txin = 0;
        m_Txout = 0;
        m_Rxin = 0;
        m_Rxout = 0;

        // enable TX
        m_bTxEnable = true;

        // NOTE: RX enabled in ISR mux
    }
    else
    {
        m_bTxEnable = false;
        DisableHardware();
    }
}

/**
 * \author      Andrew Salmon
 *
 * \date        09/18/2008
 *
 * \brief       Change UART RTS state
 *
 * \param       state - Bit_SET (asserts)/Bit_Reset (deasserts)
 * \param       Channel - serial port instance
 */
void ProcessDigitalRTSJoin(UINT8 Channel, BOOL state)
{
    SerialPort *pTmpSerialPort;

    pTmpSerialPort = GetSerialPort(Channel);

    if (pTmpSerialPort)
        pTmpSerialPort->SetRTSToDigitalJoinState(state);

    return;
}
/*
 * \author    Andrew Salmon 
 * \brief     Converts COM channel # to COM letter
 * \date      08/06/2013
 * \param     pointer to COM char 
 * \param     pointer to channel # variable 
 * \return    int
 * \retval    0 = OK or 0xff
 * \note
 **/
int GetChannelCharFromNumber(char *p, UINT8 channel)
{
    switch (channel)
        {
        case SERIAL_PORT_COMA:
            *p = 'A';
            break;
        case SERIAL_PORT_COMB:
            *p = 'B';
            break;
        case SERIAL_PORT_COMC:
            *p = 'C';
            break;
        case SERIAL_PORT_COMD:
            *p = 'D';
            break;
        case SERIAL_PORT_COME:
            *p = 'E';
            break;
        case SERIAL_PORT_COMF:
            *p = 'F';
            break;
    default:
        return 0xff;
    }
    return 0;
}
/*
 * \author    Andrew Salmon 
 * \brief     Converts COM letter to channel #
 * \date      08/06/2013
 * \param     pointer to COM char 
 * \param     pointer to channel # variable 
 * \return    int
 * \retval    0 = OK or 0xff
 * \note
 **/
int GetChannelNumberFromChar(char *p, UINT8 *channel)
{
    switch (*p)
        {
        case 'A':
        case 'a':
            *channel = SERIAL_PORT_COMA;
            break;
        case 'B':
        case 'b':
            *channel = SERIAL_PORT_COMB;
            break;
        case 'C':
        case 'c':
            *channel = SERIAL_PORT_COMC;
            break;
        case 'D':
        case 'd':
            *channel = SERIAL_PORT_COMD;
            break;
        case 'E':
        case 'e':
            *channel = SERIAL_PORT_COME;
            break;
        case 'F':
        case 'f':
            *channel = SERIAL_PORT_COMF;
            break;
    default:
        return 0xff;
    }
    return 0;
}
/**
 * \author    Andrew Salmon
 * \brief     Sends string to serial port
 * \date      04/17/2009
 * \param     pointer to command line arguments
 * \return    int
 * \retval    0 = OK or error
 * \note
 **/
INT32 SendSerialCmd(UINT32 ignore, char * cmd)
{
    char *p;
    UINT8 channel, found =0; //port
    char *endptr;
    unsigned char value;
    int k = 0, len;
    const SerialPortCfg *pSerialCfg = &SerialCfg[0];

    if (!(cmd) || !(*cmd) || (*cmd == '?'))
    {
        DmConsolePrintf("SENDSERIAL [port, string]\r");
        DmConsolePrintf("\tport - 1 of 6 possible ports (A - F)\r");
        DmConsolePrintf("\tstring size 68 max\r");
        return 0;
    }

    // decode argument string
    p = strtok(cmd, " -.,;:");
    if (!p)
    {
        DmConsolePrintf("Error parsing port byte\r");
        return -1;
    }

    if(GetChannelNumberFromChar(p, &channel)!=0)
    {
        DmConsolePrintf("Invalid port, Port [%c]\r", p);
        DmConsolePrintf("\tport - A,B,C,D,E or F\r");
        return 0;
    }

    p = strtok(NULL, 0); //end of string

    if (!p)
        return 0;

    for( int index = 0; index < NumSerialPorts; index++)
    {
        if(pSerialCfg[index].Channel == channel )
        {
            found = 1;
        }
    }

    if( found == 0 )
    {
        DmConsolePrintf("Port [%c] not supported\r", p);
        return 0;
    }
    else
    {
        len = strlen(p);

        for (k = 0; (k < len); k++)
        {
            value = p[k];
            // check to see if this character starts an escape...
            if (p[k] == '\\')
            {
                // verify that the '\' is not at the end of the string... if it is, error
                if (!(k + 2 > len))
                {
                    switch (p[k + 1])
                    {
                    case '\\':
                        //newstring[j++] = 0x5C;	//Backslash
                        value = 0x5C;
                        SerialPortReceiveTxData(channel, (char*) &value, 1);
                        k++;
                        break;

                        //Hexadecimal character
                    case 'X':
                    case 'x':
                        // Can we grab enough characters before processing the number?
                        if ((k + 4) > len)
                        {
                            SerialPortReceiveTxData(channel, (char*) &value, 1);
                        }
                        else
                        {
                            if (isxdigit(p[k + 2]) && isxdigit(p[k + 3]))
                            {
                                p[k] = '0'; //pad with '0' for format 0xXX for strtol
                         value = (unsigned char) strtol( (const char*) &p[k], &endptr, 16);
                                k += 3; // make sure to start at next character...
                         SerialPortReceiveTxData(channel, (char*)&value, 1 );
                            }
                            else
                            {
                         SerialPortReceiveTxData(channel, (char*)&value, 1 );
                            }
                        }
                        break;

                    default:
                        SerialPortReceiveTxData(channel, (char*) &value, 1);
                        break;
                    }
                }
                else
                {
                    // not part of an escape, add it to the string...
                    SerialPortReceiveTxData(channel, (char*) &value, 1);
                }
            }
            else
            {
                // not part of an escape, add it to the string...
                SerialPortReceiveTxData(channel, (char*) &value, 1);
            }
        }
    }

    return 0;

}

/**
 * \author    Andrew Salmon - partially ported from AMS-AIP
 * \brief     Configure serial port
 * \date      04/17/2009
 * \param     pointer to command line arguments
 * \return    int
 * \retval    0 = OK or error
 * \note
 **/
INT32 SetCommParameters(UINT32 ignore, char * cmd)
{
    char *p, par, charChan;
    char *endptr;
    UINT32 baud, stopbits, parity, databits;
    UINT8 index, channel, found=0, comSpec = 0;
    SerialPort *pSerialStream;
    const SerialPortCfg *pSerialCfg = &SerialCfg[0];

    if (!cmd || !(*cmd))                 // print current status
    {
       for( index = 0; index < NumSerialPorts; index++)
        {
            pSerialStream = GetSerialPort(pSerialCfg[index].Channel);
            if(GetChannelCharFromNumber(&charChan, pSerialCfg[index].Channel)!= 0)
            {
                DmConsolePrintf("Invalid Channel # %d in SerialCfg[%d]\r", pSerialCfg[index].Channel, index);
                continue;
            }

            if (pSerialStream)
            {
                pSerialStream->GetHwConfig(&baud, &stopbits, &parity, &databits);

                switch (parity)
                {
                case UART_CRESNET_PARITY_EVEN:
                    par = 'E';
                    break;
                case UART_CRESNET_PARITY_ODD:
                    par = 'O';
                    break;
                case UART_CRESNET_PARITY_NO:
                default:
                    par = 'N';
                    break;
                }

                DmConsolePrintf("Current Comspec Port %c = %d, %c,%d,%d\r",charChan,
                baud, par, databits, stopbits);
            }
            else
            {
                DmConsolePrintf("Serial Port [%c] Not Initialized\r", charChan);
            }
        }
        return 0;
    }

    if (*cmd == '?')                   // print help string
    {
        DmConsolePrintf("SERIAL [baud,parity,databits,stopbits,port]\r");
        DmConsolePrintf("\tbaud - baud rate (300,600,1200,1800,...,34800,57600,115200)\r\n \t (in dec.)\r");
        DmConsolePrintf("\tparity - single character representing parity (N,O,E)\r");
        DmConsolePrintf("\tdatabits - number of data bits (7 or 8) \r");
        DmConsolePrintf("\tstopbits - number of stop bits (1,2) \r");
        DmConsolePrintf("\tport - 1 of 8 possible ports (A - H) default port A\r");
        DmConsolePrintf("\tNo parameter - displays current setting on port A\r");
        return 0;
    }

    // decode argument string
    p = strtok(cmd, " -.,;:");
    if (!p)
    {
        DmConsolePrintf("Error parsing baud rate\r");
        return -1;
    }
    baud = (unsigned long) strtol(p, &endptr, 10);  // read baud

    switch (baud)
    {
    case 300:
        comSpec = COMSPEC_300;
        break;
    case 600:
        comSpec = COMSPEC_600;
        break;
    case 1200:
        comSpec = COMSPEC_1200;
        break;
    case 1800:
        comSpec = COMSPEC_1800;
        break;
    case 2400:
        comSpec = COMSPEC_2400;
        break;
    case 3600:
        comSpec = COMSPEC_3600;
        break;
    case 4800:
        comSpec = COMSPEC_4800;
        break;
    case 7200:
        comSpec = COMSPEC_7200;
        break;
    case 9600:
        comSpec = COMSPEC_9600;
        break;
    case 14400:
        comSpec = COMSPEC_14400;
        break;
    case 19200:
        comSpec = COMSPEC_19200;
        break;
    case 28800:
        comSpec = COMSPEC_28800;
        break;
    case 38400:
        comSpec = COMSPEC_38400;
        break;
    case 57600:
        comSpec = COMSPEC_57600;
        break;
    case 115200:
        comSpec = COMSPEC_115200;
        break;
    default:
        DmConsolePrintf("Invalid Baud Rate\r");
        DmConsolePrintf("SERIAL [baud,parity,databits,stopbits,port]\r");
        DmConsolePrintf("\tbaud - baud rate (300 - 460800) (in dec.)\r");
        return 0;
    }

    p = strtok(NULL, " -.,;:");
    if (!p)
    {
        DmConsolePrintf("Error parsing parity\r");
        return -1;
    }

    switch ( *p )// read parity
    {
    case 'E':
    case 'e':
        comSpec |= COMSPEC_PEVEN;
        break;

    case 'O':
    case 'o':
        comSpec |= COMSPEC_PODD;
        break;

    case 'N':
    case 'n':
        comSpec |= COMSPEC_PNONE;
        break;

    default:
        DmConsolePrintf("Invalid Parity\r");
        DmConsolePrintf("SERIAL [baud,parity,databits,stopbits,port]\r");
        DmConsolePrintf("\tparity - single character representing parity (N,O,E)\r");
        return 0;
    }

    p = strtok(NULL, " -.,;:");
    if (!p)
    {
        DmConsolePrintf("Error parsing data bits\r");
        return -1;
    }
    databits = (unsigned long) strtol(p, &endptr, 10);  // read data bits

    switch (databits)
    {
    case 7:
        comSpec |= COMSPEC_7BITS;
        break;
    case 8:
        comSpec |= COMSPEC_8BITS;
        break;
    default:
        DmConsolePrintf("Invalid databits\r");
        DmConsolePrintf("SERIAL [baud,parity,databits,stopbits,port]\r");
        DmConsolePrintf("\tdatabits - number of data bits (7 or 8) \r");
        return 0;
    }

    p = strtok(NULL, " -.,;:");
    if (!p)
    {
        DmConsolePrintf("Error parsing stop bits\r");
        return -1;
    }
    stopbits = (unsigned long) strtol(p, &endptr, 10);  // read stop bits

    switch (stopbits)
    {
    case 1:
        comSpec |= COMSPEC_1STOP;
        break;
    case 2:
        comSpec |= COMSPEC_2STOP;
        break;
    default:
        DmConsolePrintf("Invalid stopbits\r");
        DmConsolePrintf("SERIAL [baud,parity,databits,stopbits,port]\r");
        DmConsolePrintf("\tstopbits - number of stop bits (1,2) \r");
        return 0;
    }

    p = strtok(NULL, " -.,;:");  //get port
    if (!p)
    {
        channel = SERIAL_PORT_COMA;  //use port A for default
    }
    else
    {
        if(GetChannelNumberFromChar(p, &channel) != 0)
        {    
            DmConsolePrintf("Invalid port, Port [%c]\r", p);
            DmConsolePrintf("SERIAL [baud, parity, databits, stopbits, port]\r");
            DmConsolePrintf("\tport - A,B,C,D,E or F\r");
            return 0;
        }
    }
    
    for( index = 0; index < NumSerialPorts; index++)
    {
        if(pSerialCfg[index].Channel == channel )
        {
            found = 1;
        }
    }

    if( found == 0 )
    {
        DmConsolePrintf("Port [%c] not supported\r", p);
        return 0;
    }

    DmConsolePrintf("\r");

    pSerialStream = GetSerialPort(channel);

    if (pSerialStream)
    {
        pSerialStream->SetCommParameters(comSpec);
    }
    return 0;
}
/*
 * \author    Andrew Salmon - partially ported from AMS-AIP
 * \brief     Configure serial handshake
 * \date      04/20/2009
 * \param     pointer to command line arguments
 * \return    int
 * \retval    0 = OK or error
 * \note
 **/
INT32 SetUartHWHandshake(UINT32 ignore, char *cmd)
{
    char *p, charChan;
    SerialPort *pSerialStream;
    UINT8 channel, index, found =0;
    const SerialPortCfg *pSerialCfg = &SerialCfg[0];

    if (cmd && *cmd == '?')                   // print help strilng
    {
        DmConsolePrintf("RTSCTS [port, ON | OFF]\r");
        DmConsolePrintf("\tNo parameter - displays current setting\r");
        return 0;
    }

    // decode argument string
    p = strtok(cmd, " -.,;:");
    if (!p)
    {
        for( index = 0; index < NumSerialPorts; index++)
        {
            pSerialStream = GetSerialPort(pSerialCfg[index].Channel);
            if(GetChannelCharFromNumber(&charChan, pSerialCfg[index].Channel)!= 0)
            {
                DmConsolePrintf("Invalid Channel # %d in SerialCfg[%d]\r", pSerialCfg[index].Channel, index);
                continue;
            }

            if (pSerialStream)
            {
                DmConsolePrintf("HW handshake Port %c: %s\r", charChan, pSerialStream->ReturnHwHandShake() ? "ON" : "OFF");
            }
            else
            {
                DmConsolePrintf("Serial Port %c Not Initialized", charChan);
            }
        }
        return 0;
    }
    if(GetChannelNumberFromChar(p, &channel)!=0)
    {
        DmConsolePrintf("Invalid port, Port [%c]\r", p);
        DmConsolePrintf("\tport - A,B,C,D,E or F\r");
        return 0;
    }
    charChan = *p;                   //save letter channel
    // decode argument string
    p = strtok(NULL, " -.,;:");    //get port

    if (!p)    //no argument
    {
        for( index = 0; index < NumSerialPorts; index++)
        {
            if(pSerialCfg[index].Channel == channel )
            {
                found = 1;
            }
        }
    
        if( found == 0 )
        {
            DmConsolePrintf("Port [%c] not supported\r", p);
            return 0;
        }

        pSerialStream = GetSerialPort(channel);
        if (pSerialStream)
        {
          DmConsolePrintf("HW handshake Port %c: %s\r", charChan, pSerialStream->ReturnHwHandShake() ? "ON" : "OFF");
        }
        else
        {
            DmConsolePrintf("Serial Port %c Not Initialized", charChan);
        }
        return 0;
    }

    LocalConvertToUpper(p);           // convert parameter to uppercase

    for( index = 0; index < NumSerialPorts; index++)
    {
        if(pSerialCfg[index].Channel == channel )
        {
            found = 1;
        }
    }

    if( found == 0 )
    {
        DmConsolePrintf("Port [%c] not supported\r", p);
        return 0;
    }

    pSerialStream = GetSerialPort(channel);

    if (pSerialStream)
    {
        if (strcmp(p, "ON") == 0)
        {
            //Enable Hw Handshake
            pSerialStream->EnableHwHandShake(true);
        }
        else if (strcmp(p, "OFF") == 0)
        {
            //Disable Hw Handshake
            pSerialStream->EnableHwHandShake(false);

        }
        else
        {
            DmConsolePrintf("Unknown option %c.\r", p);
            return -1;
        }
    }
    else
    {
        DmConsolePrintf("Serial Port %c Not Initialized", charChan);
    }

    return 0;
}

void SerialPort::PrintDebug()
{
	UINT32 bps, stopbits, parity, wlength;
	UINT16 pacing;
	char parity_char;
	DecodeComspec(m_NewComSpec,bps,stopbits,parity,wlength,pacing);
	//DmConsolePrintf("  Instance = %d\r", inst);
	switch (parity)
	{
	case UART_CRESNET_PARITY_ODD:
		parity_char = 'O';
		break;
	case UART_CRESNET_PARITY_EVEN:
		parity_char = 'E';
		break;
	case UART_CRESNET_PARITY_NO:
		parity_char = 'N';
		break;
	}
	DmConsolePrintf("  Comspec = %d %d%c%d\r", 
					bps, wlength, parity_char, stopbits);
	DmConsolePrintf("  TxEnable = %d\r", m_bTxEnable);
}


/**
 * \author      Chris Merck
 * \date        2 JULY 2013
 *
 * \return      INT32
 * \retval      0 success, no output written
 * \retval      1 success, raw data written to pData
 * \retval      2 success, ascii output written to pData
 * \retval      <0 failure
 *
 * \brief       Device driver entry point for this device
 *
 * \param       Channel - which device if there are multiple of 
 *              same type
 * \param       cmd - particular command
 * \param       pData - associated data pointer
 * \param       pDataBytes - associated bytes of data
 */
INT32 SerialPortDeviceProcessInput(UINT32 Channel, UINT32 cmd, UINT8 * pData, UINT32 * pDataBytes)
{
    SerialPort * pSerial = NULL;

    //Verify the data
    if (*pDataBytes == 0)
        return 0;

    //Verify the instance number
    if ( pData[0] >= NumSerialPorts )
    {
        DmConsolePrintf("Invalid SerialPort %d\r", pData[0]);
        return 0;
    }

    //Grab the SerialPort class
    pSerial = GetSerialPort(pData[0]);

    switch(cmd)
    {
        case DEVICE_PRINT:
            //Print the debug information
            pSerial->PrintDebug();
            break;

        case DEVICE_SET:
		{
			UINT8 pComspecPacket[6];
		    //  format:
			//    <80h+chan> <00> <comspec_low> <comspec_hi> <pacing> <max len>
			//    <comspec_low>, <comspec_hi> are build with the below table.
			//    <pacing>:  in milliseconds (31 ms max)
			//    <max len>: max string length, 246 max, 0 means use default
			pComspecPacket[0] = 0x80 + pData[0]; 
			pComspecPacket[1] = 0x00;
			EncodeComspec(pComspecPacket+2,9600,1,0,8,0);
			pSerial->ReceiveComspec(pComspecPacket);
			break;
		}
        case DEVICE_GET:
        case DEVICE_DETECT:
        case DEVICE_SELFTEST:
        case DEVICE_INIT:
			break;
        case DEVICE_TX:
		    pSerial->ReceiveTxData("hello",5);
			break;
        case DEVICE_RX:
        case DEVICE_SET_DEBUG:
        case DEVICE_GET_DEBUG:
        default:
            break;
    }
    return 0;
}
